# Admin Authentication System

Complete backend implementation for admin login system with JWT authentication.

## Features

- ✅ Secure password hashing with bcrypt
- ✅ JWT token-based authentication
- ✅ Admin model with role-based access control
- ✅ Login endpoint with validation
- ✅ Protected routes with authentication middleware
- ✅ Admin seeder for initial setup
- ✅ Comprehensive error handling
- ✅ Request logging integration

## API Endpoints

### Public Routes

#### Login

```http
POST /api/auth/login
Content-Type: application/json

{
  "email": "admin@samaa.com",
  "password": "Admin@123"
}
```

**Success Response (200):**

```json
{
  "success": true,
  "message": "Login successful",
  "data": {
    "admin": {
      "_id": "...",
      "email": "admin@samaa.com",
      "name": "Admin User",
      "role": "super_admin",
      "isActive": true,
      "lastLogin": "2024-01-01T00:00:00.000Z",
      "createdAt": "2024-01-01T00:00:00.000Z",
      "updatedAt": "2024-01-01T00:00:00.000Z"
    },
    "token": "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9..."
  }
}
```

**Error Response (401):**

```json
{
  "success": false,
  "error": "Invalid email or password"
}
```

### Protected Routes (Require Authentication)

#### Get Current Admin Profile

```http
GET /api/auth/me
Authorization: Bearer <token>
```

**Success Response (200):**

```json
{
  "success": true,
  "data": {
    "admin": {
      "_id": "...",
      "email": "admin@samaa.com",
      "name": "Admin User",
      "role": "super_admin",
      "isActive": true
    }
  }
}
```

#### Logout

```http
POST /api/auth/logout
Authorization: Bearer <token>
```

**Success Response (200):**

```json
{
  "success": true,
  "message": "Logout successful"
}
```

## Setup Instructions

### 1. Install Dependencies

```bash
npm install
```

### 2. Configure Environment Variables

Copy `.env.example` to `.env` and update the values:

```env
# JWT Configuration
JWT_SECRET=your-secret-key-here-change-in-production
JWT_EXPIRE=7d

# Admin Seeder Configuration
ADMIN_EMAIL=admin@samaa.com
ADMIN_PASSWORD=Admin@123
ADMIN_NAME=Admin User
ADMIN_ROLE=super_admin
```

### 3. Seed Admin User

```bash
npm run seed:admin
```

This will create the default admin user with the credentials specified in your `.env` file.

**Output:**

```
✅ Admin created successfully!
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
📧 Email: admin@samaa.com
🔑 Password: Admin@123
👤 Name: Admin User
🎭 Role: super_admin
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

⚠️  Please change the default password after first login!
```

### 4. Start Server

```bash
npm run dev
```

## Usage Examples

### Frontend Integration

#### Login Request

```javascript
const login = async (email, password) => {
  try {
    const response = await fetch("http://localhost:3000/api/auth/login", {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
      },
      body: JSON.stringify({ email, password }),
    });

    const data = await response.json();

    if (data.success) {
      // Store token in localStorage or cookie
      localStorage.setItem("token", data.data.token);
      localStorage.setItem("admin", JSON.stringify(data.data.admin));
      return data;
    } else {
      throw new Error(data.error);
    }
  } catch (error) {
    console.error("Login error:", error);
    throw error;
  }
};
```

#### Authenticated Request

```javascript
const getProfile = async () => {
  try {
    const token = localStorage.getItem("token");
    const response = await fetch("http://localhost:3000/api/auth/me", {
      method: "GET",
      headers: {
        Authorization: `Bearer ${token}`,
      },
    });

    const data = await response.json();
    return data;
  } catch (error) {
    console.error("Get profile error:", error);
    throw error;
  }
};
```

## Security Features

1. **Password Hashing**: Passwords are hashed using bcrypt with salt rounds of 12
2. **JWT Tokens**: Secure token-based authentication
3. **Token Expiration**: Configurable token expiration (default: 7 days)
4. **Password Exclusion**: Passwords are never returned in API responses
5. **Account Status Check**: Inactive accounts cannot login
6. **Input Validation**: Email format and required fields validation
7. **Error Messages**: Generic error messages to prevent user enumeration

## File Structure

```
src/
├── models/
│   └── Admin.js              # Admin model with password hashing
├── controllers/
│   └── authController.js     # Login, logout, getMe handlers
├── routes/
│   └── authRoutes.js         # Authentication routes
├── middleware/
│   └── auth.js               # Authentication & authorization middleware
└── seeders/
    └── adminSeeder.js        # Admin user seeder
```

## Model Schema

```javascript
{
  email: String (required, unique, lowercase, validated)
  password: String (required, minlength: 6, hashed)
  name: String (required)
  role: String (enum: ['admin', 'super_admin'], default: 'admin')
  isActive: Boolean (default: true)
  lastLogin: Date
  createdAt: Date (auto)
  updatedAt: Date (auto)
}
```

## Error Handling

All errors are properly handled and logged:

- **400**: Missing required fields
- **401**: Invalid credentials or missing token
- **403**: Account deactivated or insufficient permissions
- **500**: Server errors

## Notes

- Change the default admin password after first login
- Use a strong JWT_SECRET in production
- The seeder will skip if admin already exists
- All login attempts are logged for security monitoring
